<?php

namespace Drupal\Tests\xray_audit\Kernel;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\NodeType;
use Drupal\KernelTests\KernelTestBase;

/**
 * Base class for Xray Audit kernel tests.
 *
 * Provides common functionality for kernel tests including module installation,
 * entity setup, and shared configuration.
 *
 * @codingStandardsIgnoreFile
 */
abstract class XrayAuditKernelTestBase extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'filter',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install necessary schemas.
    $this->installEntitySchema('user');
    $this->installEntitySchema('node');

    // KernelTestBase::bootKernel() sets a global override for system.file
    // which can cause schema validation errors in strict modes (e.g., CI).
    // Remove the global override before installing config.
    unset($GLOBALS['config']['system.file']);

    $this->installConfig(['system', 'field', 'node', 'user']);

    // Set the default file scheme to public.
    \Drupal::configFactory()->getEditable('system.file')->set('default_scheme', 'public')->save();

    // Additional common setup can go here.
  }

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container) {
    parent::register($container);

    // Register the public stream wrapper for file operations.
    // Required in CI environments where stream wrappers need explicit registration.
    $container->register('stream_wrapper.public', 'Drupal\Core\StreamWrapper\PublicStream')
      ->addTag('stream_wrapper', ['scheme' => 'public']);

    // Register the private stream wrapper for file operations.
    $container->register('stream_wrapper.private', 'Drupal\Core\StreamWrapper\PrivateStream')
      ->addTag('stream_wrapper', ['scheme' => 'private']);
  }

  /**
   * {@inheritdoc}
   */
  protected function setUpFilesystem() {
    parent::setUpFilesystem();

    // Set up private and temporary file directories.
    $private_file_directory = $this->siteDirectory . '/private';
    $temp_file_directory = $this->siteDirectory . '/temp';

    // Create the directories.
    mkdir($private_file_directory, 0775);
    mkdir($temp_file_directory, 0775);

    // Configure file paths using setSetting() method.
    $this->setSetting('file_private_path', $private_file_directory);
    $this->setSetting('file_temp_path', $temp_file_directory);
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
    // Common cleanup for all kernel tests can go here.
  }

  /**
   * Creates a test content type.
   *
   * @param string $type_id
   *   The content type machine name.
   * @param string $label
   *   The content type label.
   *
   * @return \Drupal\node\Entity\NodeType
   *   The created content type.
   */
  protected function createContentType(string $type_id, ?string $label = NULL): NodeType {
    $type = NodeType::create([
      'type' => $type_id,
      'name' => $label ?? $type_id,
      'new_revision' => TRUE,
    ]);
    $type->save();
    return $type;
  }

  /**
   * Creates a test field on an entity type.
   *
   * @param string $field_name
   *   The field name.
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle.
   * @param string $field_type
   *   The field type.
   * @param array $storage_settings
   *   Optional field storage settings.
   * @param array $field_settings
   *   Optional field instance settings.
   */
  protected function createTestField(
    string $field_name,
    string $entity_type,
    string $bundle,
    string $field_type,
    array $storage_settings = [],
    array $field_settings = [],
  ): void {
    // Create field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => $entity_type,
      'type' => $field_type,
      'settings' => $storage_settings,
    ]);
    $field_storage->save();

    // Create field instance.
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $bundle,
      'label' => $field_name,
      'settings' => $field_settings,
    ]);
    $field->save();
  }

}
