<?php

namespace Drupal\yayb\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Defines the Notification Settings Form.
 */
class YaybSettingsForm extends ConfigFormBase {

  use AutowireTrait;

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'yayb.settings';

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The account proxy interface.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    protected TypedConfigManagerInterface $typedConfigManager,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly AccountProxyInterface $currentUser,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'yayb_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildForm(array $form, FormStateInterface $form_state) {

    $config = $this->config(static::SETTINGS);

    $datetime = $config->get('yayb_expiration') ?? NULL;
    if (NULL !== $datetime) {
      $datetime = new DrupalDateTime($datetime);
    }

    $form[] = [
      '#markup' => $this->t('<div><strong>Note: Page and block caching may affect visibility.</strong></div>'),
    ];
    $form['yayb_toggle'] = [
      '#type' => 'radios',
      '#title' => $this->t('Display Options'),
      '#default_value' => $config->get('yayb_toggle') ?? 0,
      '#options' => [
        1 => $this->t('Show Alert Box'),
        0 => $this->t('Hide Alert Box'),
      ],
      '#description' => $this->t('Current message visibility. Must be on even when using expiration time below.'),
    ];
    $form['yayb_expiration'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Expiration Date'),
      '#default_value' => $datetime,
      '#description' => $this->t('If this field is blank, the message will be displayed until disabled manually.'),
    ];
    $form['yayb_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#default_value' => $config->get('yayb_title') ?? $this->t('Important Message'),
      '#description' => $this->t('Title for message block.'),
    ];
    $form['yayb_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#default_value' => $config->get('yayb_message') ?? NULL,
      '#description' => $this->t('Message to display to users.'),
    ];
    $form['yayb_filter'] = [
      '#type' => 'select',
      '#title' => $this->t('Text Format'),
      '#options' => $this->getFilterTypes(),
      '#default_value' => $config->get('yayb_filter') ?? 'plain_text',
      '#description' => $this->t('Text filter to apply to message.'),
    ];
    $form['yayb_severity'] = [
      '#type' => 'radios',
      '#title' => $this->t('Severity'),
      '#options' => [
        'critical' => $this->t('<span class="yayb-critical">Critical</span>'),
        'major' => $this->t('<span class="yayb-major">Major</span>'),
        'important' => $this->t('<span class="yayb-important">Important</span>'),
        'notice' => $this->t('<span class="yayb-notice">Notice</span>'),
        'info' => $this->t('<span class="yayb-info">Info</span>'),
      ],
      '#default_value' => $config->get('yayb_severity') ?? 'important',
      '#description' => $this->t('Set the severity level of the message. (Use css styling in theme to alter the display.)'),
    ];

    $form['#attached']['library'][] = 'yayb/yayb-styling';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function submitForm(array &$form, FormStateInterface $form_state) {

    // Retrieve the configuration.
    $this->configFactory->getEditable(static::SETTINGS)
      ->set('yayb_toggle', $form_state->getValue('yayb_toggle'))
      ->set('yayb_expiration', $form_state->getValue('yayb_expiration')?->format('Y-m-d H:i:s'))
      ->set('yayb_title', $form_state->getValue('yayb_title'))
      ->set('yayb_message', $form_state->getValue('yayb_message'))
      ->set('yayb_filter', $form_state->getValue('yayb_filter'))
      ->set('yayb_severity', $form_state->getValue('yayb_severity'))
      ->save();

    parent::submitForm($form, $form_state);

    // Clear caches to retstore block visibility.
    drupal_flush_all_caches();
  }

  /**
   * Get available filter types.
   */
  private function getFilterTypes(): array {
    $filter_types = [];
    foreach ($this->entityTypeManager->getStorage('filter_format')->loadMultiple() as $id => $format) {
      if ($format->access('use', $this->currentUser)) {
        $filter_types[$id] = $format->label();
      }
    }
    return $filter_types;
  }

}
