<?php

/**
 *  @author YooMoney <cms@yoomoney.ru>
 *  @copyright © 2025 "YooMoney", NBСO LLC
 *  @license  https://yoomoney.ru/doc.xml?id=527052
 */

namespace Drupal\yookassa\Controller;

use Drupal;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\yookassa\Helpers\YooKassaLoggerHelper;
use Drupal\yookassa\Helpers\YooKassaPaymentMethodHelper;
use Drupal\yookassa\Plugin\Commerce\PaymentGateway\YooKassa;
use Exception;
use Drupal\yookassa\Oauth\YooKassaClientFactory;
use Drupal\yookassa\Oauth\YooKassaWebhookSubscriber;
use Drupal\yookassa\Oauth\YooKassaOauth;
use JsonException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Контроллер для работы с oauth
 */
class YooKassaOauthController extends ControllerBase
{
    /**
     * Делает запрос к OAuth приложению для получения ссылки на авторизацию.
     *
     * @param Request $request Модель запроса
     *
     * @return AjaxResponse
     * @throws JsonException
     */
    public function getOauthUrl(Request $request): AjaxResponse
    {
        $machineName = $request->get('name');
        $formData = $request->get('form');

        $response = new AjaxResponse();

        if (!$request->isXmlHttpRequest()) {
            $response->setData(['status' => 'error', 'error' => 'Unknown', 'code' => 'unknown']);
            return $response;
        }

        try {
            $config = YooKassaPaymentMethodHelper::check($machineName)
                ? Drupal::configFactory()->getEditable(
                    'commerce_payment.commerce_payment_gateway.' . $machineName
                )->getOriginal('configuration')
                : YooKassaPaymentMethodHelper::savePaymentMethod($formData, $machineName)?->toArray()['configuration'];
        } catch (Exception $e) {
            $this->errorExit('Error while saving payment method: ' . $e->getMessage(), $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if (!$config) {
            $this->errorExit('Payment method configuration is empty');
        }

        $kassaLogger = $this->getKassaLogger($config['shop_id'] ?? null);
        $kassaLogger->sendHeka(['oauth.process.init']);

        try {
            $oauth = new YooKassaOauth($machineName);
            $result = $oauth->generateOauthUrl();
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.process.fail']);
            $this->errorExit('Exception: ' . $e->getMessage(), $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if ($result['code'] !== Response::HTTP_OK) {
            $kassaLogger->sendHeka(['oauth.process.fail']);
            $this->errorExit('Got error while getting OAuth link.');
        }

        $body = json_decode($result['response'], true, 512, JSON_THROW_ON_ERROR);

        if (!isset($body['oauth_url'])) {
            $kassaLogger->sendHeka(['oauth.process.fail']);
            $error = $body['error'] ?? 'OAuth URL not found';
            $this->errorExit('Got error while getting OAuth link. Response body: ' . json_encode($body, JSON_THROW_ON_ERROR), $error);
        }

        $response->setData(['oauth_url' => $body['oauth_url']]);
        $kassaLogger->sendHeka(['oauth.process.success']);

        return $response;
    }

    /**
     * Функция обработки ajax запроса на получение OAuth токена через OAuth-приложение.
     *
     * @param Request $request Модель запроса
     *
     * @return AjaxResponse
     * @throws Exception
     */
    public function getOauthToken(Request $request): AjaxResponse
    {
        $machineName = $request->get('name');
        $notifyUrl = $request->get('notifyUrl');

        $response = new AjaxResponse();

        if (!$request->isXmlHttpRequest()) {
            $response->setData(['status' => 'error', 'error' => 'Unknown', 'code' => 'unknown']);
            return $response;
        }

        $oauth = new YooKassaOauth($machineName);
        $config = $oauth->paymentMethodEditConfig->getOriginal('configuration');
        $kassaLogger = $this->getKassaLogger($config['shop_id'] ?? null);
        $kassaLogger->sendHeka(['oauth.callback.init']);

        try {
            $kassaLogger->sendHeka(['oauth.get-token.init']);
            $result = $oauth->generateOauthToken();
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $this->errorExit('Exception: ' . $e->getMessage(), $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if ($result['code'] === Response::HTTP_UNPROCESSABLE_ENTITY) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $body = json_decode($result['response'], true, 512, JSON_THROW_ON_ERROR);
            $error = $body['error'] ?? 'Access token not found';
            $this->errorExit($error, 'Авторизация не пройдена');
        }

        if ($result['code'] !== Response::HTTP_OK) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $this->errorExit('Got error while getting OAuth token.');
        }

        $body = json_decode($result['response'], true, 512, JSON_THROW_ON_ERROR);

        if (!isset($body['access_token'])) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $error = $body['error'] ?? 'Access token not found';
            $this->errorExit('Got error while getting OAuth token. Key access_token not found. Response body: ' . json_encode($body, JSON_THROW_ON_ERROR), $error);
        }

        if (!isset($body['expires_in'])) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $error = $body['error'] ?? 'Expires_in parameter not found';
            $this->errorExit('Got error while getting OAuth token. Key expires_in not found. Response body: ' . json_encode($body, JSON_THROW_ON_ERROR), $error);
        }

        $token = $config['access_token'] ?? null;

        $config['access_token'] = $body['access_token'];
        $config['token_expires_in'] = $body['expires_in'];
        $config['notification_url'] = $notifyUrl;

        $oauth->saveConfigurationPayment([
            'configuration.access_token' => $body['access_token'],
            'configuration.token_expires_in' => $body['expires_in'],
            'configuration.notification_url' => $notifyUrl,
            'status' => 1
        ]);
        $kassaLogger->sendHeka(['oauth.get-token.success']);

        if ($token) {
            try {
                $kassaLogger->sendHeka(['token.revoke.init']);
                $oauth->revokeOldToken($token);
                $kassaLogger->sendHeka(['token.revoke.success']);
            } catch (Exception $e) {
                $kassaLogger->sendHeka(['oauth.callback.fail', 'token.revoke.fail']);
                $this->log('Error when revoking old token: ' . $e->getMessage(), 'error');
            }
        }

        try {
            $client = YooKassaClientFactory::getYooKassaClient($config);
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail']);
            $this->errorExit('Error when creating a client: ' . $e->getMessage(), $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        try {
            $kassaLogger->sendHeka(['oauth.get-shop.init']);
            $oauth->saveShopInfoByOauth();
            $kassaLogger->sendHeka(['oauth.get-shop.success']);
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-shop.fail']);
            $this->errorExit('Error to get information about shop: ' . $e->getMessage(), $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        try {
            $kassaLogger->sendHeka(['webhooks.subscribe.init']);
            YooKassaWebhookSubscriber::subscribe($client, $config);
            $kassaLogger->sendHeka(['webhooks.subscribe.success']);
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'webhooks.subscribe.fail']);
            $this->errorExit('Error occurred during creating webhooks: ' . $e->getMessage(), $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $url = Url::fromRoute('entity.commerce_payment_gateway.edit_form', ['commerce_payment_gateway' => $machineName])->toString();

        $response->setData(['url' => $url]);
        $kassaLogger->sendHeka(['oauth.callback.success']);
        return $response;
    }

    /**
     * Вывод сообщений об ошибках и остановка скрипта.
     *
     * @param string $errorLog Сообщение для записи в лог
     * @param string|null $errorFront Сообщение для вывода на фронт пользователю
     * @param int $code Код статуса
     * @return void
     */
    private function errorExit(string $errorLog, ?string $errorFront = null, int $code = 502): void
    {
        $errorFront = $errorFront ?: $errorLog;
        $this->log('Error: ' . $errorLog, 'error');
        echo json_encode(['error' => $errorFront]);
        exit($code);
    }

    /**
     * Создает запись в логе.
     *
     * @param string $message Сообщение
     * @param string $type Тип сообщение
     *
     * @return void
     */
    private function log(string $message, string $type = 'info'): void
    {
        Drupal::logger('yookassa')->$type($message);
    }

    /**
     * Проверка на существование записи платежного шлюза в БД по машинному имени.
     *
     * @param Request $request Модель запроса
     *
     * @return AjaxResponse
     */
    public function checkPaymentMethod(Request $request): AjaxResponse
    {
        $machineName = $request->get('name');
        $response = new AjaxResponse();
        if (!$machineName) {
            $response->setData([
                'error' => $this->t('The "Machine Name" field can\'t be empty')
            ]);
            return $response;
        }

        $response->setData([
            'error' => YooKassaPaymentMethodHelper::check($machineName)
                ? $this->t('The "@machineName" payment gateway already exists. Please enter a different name', ['@machineName' => $machineName])
                : null
        ]);

        return $response;
    }

    /**
     * Генерирует новый URL для уведомлений с новым машинным именем.
     *
     * @param Request $request Модель запроса
     *
     * @return AjaxResponse
     */
    public function generateNotificationUrl(Request $request): AjaxResponse
    {
        $machineName = $request->get('name');
        $response = new AjaxResponse();
        $response->setData(['url' => YooKassa::generateNotificationUrl($machineName)]);
        return $response;
    }

    /**
     * @param string|null $shopId
     * @return YooKassaLoggerHelper
     */
    public function getKassaLogger(?string $shopId): YooKassaLoggerHelper
    {
        return new YooKassaLoggerHelper($shopId);
    }
}
