<?php

namespace Drupal\yookassa\Helpers;

use Exception;
use Drupal;
use RuntimeException;

class YooKassaLoggerHelper
{
    /**
     * OAUTH CMS URL
     */
    public const OAUTH_CMS_URL = 'https://yookassa.ru/integration/oauth-cms';

    /**
     * @var string|null
     */
    private ?string $shopId;

    /**
     * @param string|null $shopId
     */
    public function __construct(?string $shopId = null)
    {
        $this->shopId = $shopId;
    }

    /**
     * @return string|null
     */
    public function getShopId(): ?string
    {
        return $this->shopId;
    }

    /**
     * @param string|null $shopId
     * @return YooKassaLoggerHelper
     */
    public function setShopId(?string $shopId): self
    {
        $this->shopId = $shopId;
        return $this;
    }

    /**
     * @param mixed $data
     * @return bool
     */
    public function sendMetric(mixed $data): bool
    {
        $parameters = [
            'cms' => 'drupal10',
            'host' => $_SERVER['HTTP_HOST'],
            'shop_id' => $this->shopId,
        ];

        $options = [
            CURLOPT_URL => self::OAUTH_CMS_URL . '/metric/drupal',
            CURLOPT_POSTFIELDS => json_encode(array_merge($data, $parameters)),
        ];

        try {
            $this->makeRequest($options);
        } catch (Exception $e) {
            Drupal::logger('yookassa')->error('Failed to send metric: ' . $e->getMessage());
            return false;
        }

        return true;
    }

    /**
     * @param array $metrics
     * @return void
     */
    public function sendHeka(array $metrics): void
    {
        $this->sendMetric([
            'metric_heka' => $metrics
        ]);
    }

    /**
     * @param string $type
     * @param array $metrics
     * @return void
     */
    public function sendBI(string $type, array $metrics): void
    {
        $this->sendMetric([
            'metric_bi' => [
                'type' => $type,
                'data' => $metrics
            ]
        ]);
    }

    /**
     * @param string $message
     * @param array $context
     * @param array $metrics
     * @return void
     */
    public function sendAlertLog(string $message, array $context = [], array $metrics = []): void
    {
        if (!empty($context['exception'])) {
            $exception = $context['exception'];
            if ($exception instanceof Exception) {
                $context['exception'] = [
                    'code' => $exception->getCode(),
                    'message' => $exception->getMessage(),
                    'file' => $exception->getFile() . ':' . $exception->getLine(),
                    'trace' => $exception->getTraceAsString(),
                ];
            }
        }
        $data = [
            'metric_app' => [
                'level' => 'alert',
                'message' => $message,
                'context' => $context,
            ]
        ];
        if (!empty($metrics)) {
            $data['metric_heka'] = $metrics;
        }
        $this->sendMetric($data);
    }

    /**
     * Выполняет запрос с полученными параметрами
     *
     * @param array $options - массив curl опций
     * @return void
     * @throws Exception
     */
    private function makeRequest(array $options): void
    {
        $optionsConst = [
            CURLOPT_HTTPHEADER => ['Content-Type:application/json;charset=utf-8'],
            CURLOPT_POST => 1,
            CURLOPT_RETURNTRANSFER => 1
        ];
        $options = $optionsConst + $options;
        $ch = curl_init();
        curl_setopt_array($ch, $options);
        $result = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status !== 200) {
            throw new RuntimeException(
                'Response status code is not 200. Code: ' . $status . ' Response: ' . $result
            );
        }
    }
}
