<?php

namespace Drupal\yookassa\Helpers;

use Drupal;
use Drupal\commerce_payment\Entity\Payment;
use Drupal\yookassa\Oauth\YooKassaClientFactory;
use Drupal\yookassa\Plugin\Commerce\PaymentGateway\YooKassa;
use Exception;
use YooKassa\Model\Notification\AbstractNotification;
use YooKassa\Model\Notification\NotificationEventType;
use YooKassa\Model\Payment\PaymentInterface;
use YooKassa\Model\Payment\PaymentStatus;
use YooKassa\Request\Payments\CreateCaptureRequest;
use YooKassa\Client;

class YooKassaNotificationHelper
{
    /** @var YooKassa */
    private YooKassa $kassa;

    /** @var YooKassaDatabaseHelper */
    private YooKassaDatabaseHelper $databaseHelper;

    /**
     * @param YooKassa $kassa
     */
    public function __construct(YooKassa $kassa)
    {
        $this->kassa = $kassa;
        $this->databaseHelper = new YooKassaDatabaseHelper();
    }

    /**
     * Обработка уведомления от кассы
     *
     * @param AbstractNotification $notificationObj
     * @param PaymentInterface $paymentInfo
     * @param Payment $payment
     * @return bool
     * @throws Exception
     */
    public function processNotification(AbstractNotification $notificationObj, PaymentInterface $paymentInfo, Payment $payment): bool
    {
        try {
            $apiClient = YooKassaClientFactory::getYooKassaClient($this->kassa->config);

            if ($notificationObj->getEvent() === NotificationEventType::PAYMENT_SUCCEEDED
                && $paymentInfo->getStatus() === PaymentStatus::SUCCEEDED
            ) {
                $this->handleSucceededPayment($payment, $paymentInfo);
                return true;
            }

            if ($notificationObj->getEvent() === NotificationEventType::PAYMENT_WAITING_FOR_CAPTURE
                && $paymentInfo->getStatus() === PaymentStatus::WAITING_FOR_CAPTURE
            ) {
                return $this->handleWaitingForCapture($apiClient, $paymentInfo, $payment);
            }

            if ($notificationObj->getEvent() === NotificationEventType::PAYMENT_CANCELED
                && $paymentInfo->getStatus() === PaymentStatus::CANCELED
            ) {
                $this->handleCanceledPayment();
                return true;
            }

            if ($paymentInfo->getStatus() === PaymentStatus::PENDING) {
                $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'pending');
                return true;
            }

            return false;
        } catch (Exception $e) {
            $this->getLog('Notification processing error: ' . $e->getMessage(), 'error');
            return false;
        }
    }

    /**
     * Обработка уведомления возвращение на страницу с заказом после оплаты
     *
     * @param PaymentInterface $paymentInfoResponse
     * @param Payment $payment
     * @return bool
     */
    public function processReturn(PaymentInterface $paymentInfoResponse, Payment $payment): bool
    {
        try {
            if ($paymentInfoResponse->getStatus() === PaymentStatus::SUCCEEDED) {
                $this->tryToUpdatePaymentStatus($payment, $paymentInfoResponse->getStatus(), 'completed');
                return true;
            }

            if ($paymentInfoResponse->getStatus() === PaymentStatus::PENDING && $paymentInfoResponse->getPaid()) {
                $this->tryToUpdatePaymentStatus($payment, $paymentInfoResponse->getStatus(), 'pending');
                return true;
            }

            if ($paymentInfoResponse->getStatus() === PaymentStatus::CANCELED) {
                return false;
            }

            $this->getLog('Wrong payment status: ' . $paymentInfoResponse->getStatus(), 'error');
            return false;
        } catch (Exception $e) {
            $this->getLog('Return processing error: ' . $e->getMessage(), 'error');
            return false;
        }
    }

    /**
     * Обработка уведомления об успешной оплате
     *
     * @param Payment $payment
     * @param PaymentInterface $paymentInfo
     * @return void
     * @throws Exception
     */
    private function handleSucceededPayment(Payment $payment, PaymentInterface $paymentInfo): void
    {
        $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'completed');

        $stat = $this->databaseHelper->getSuccessPaymentStat();
        if (!empty($stat)) {
            $host = $this->prepareHost(Drupal::request()->getHost());
            $this->sendHekaMetrics($host, $stat);
        }
    }

    /**
     * Обработка уведомления об ожидании подтверждения оплаты
     *
     * @param Client $apiClient
     * @param PaymentInterface $paymentInfo
     * @param Payment $payment
     * @return bool
     */
    private function handleWaitingForCapture(Client $apiClient, PaymentInterface $paymentInfo, Payment $payment): bool
    {
        $captureRequest = CreateCaptureRequest::builder()
            ->setAmount($paymentInfo->getAmount())
            ->build();

        $captureResponse = $apiClient->capturePayment($captureRequest, $paymentInfo->getId());
        $this->getLog('Payment info after capture: ' . json_encode($captureResponse));

        if ($captureResponse->status === PaymentStatus::SUCCEEDED) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'completed');
        } elseif ($captureResponse->status === PaymentStatus::CANCELED) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'canceled');
        }

        $this->getKassaLogger()->sendHeka([
            'shop.'.$this->getKassaLogger()->getShopId().'.payment.waiting_for_capture'
        ]);

        return true;
    }

    /**
     * Обработка уведомления об отмене оплаты
     *
     * @return void
     */
    private function handleCanceledPayment(): void
    {
        $this->getKassaLogger()->sendHeka([
            'shop.'.$this->getKassaLogger()->getShopId().'.payment.canceled'
        ]);
    }

    /**
     * Подготовка хоста к отправке метрик в heka
     *
     * @param string $host
     * @return string
     */
    private function prepareHost(string $host): string
    {
        return str_replace(['http://', 'https://', '.', '/', ':'], '', $host);
    }

    /**
     * Отправка метрик в heka
     *
     * @param string $host
     * @param array $stat
     * @return void
     */
    private function sendHekaMetrics(string $host, array $stat): void
    {
        $this->getKassaLogger()->sendHeka([
            'shop.' . $this->getKassaLogger()->getShopId() . '.payment.succeeded',
            'shop.' . $this->getKassaLogger()->getShopId() . '.host.' . $host . '.payment-count' => [
                'metric_type' => "gauges",
                'metric_count' => $stat['count']
            ],
            'shop.' . $this->getKassaLogger()->getShopId() . '.host.' . $host . '.payment-total' => [
                'metric_type' => "gauges",
                'metric_count' => $stat['total']
            ],
        ]);
    }

    /**
     * Обновление статуса платежа
     *
     * @param Payment $payment
     * @param string $remoteState
     * @param string $state
     * @return void
     */
    private function tryToUpdatePaymentStatus(Payment $payment, string $remoteState, string $state): void
    {
        try {
            $this->getKassaLogger()->sendHeka(['order-status.change.init']);
            $payment->setRemoteState($remoteState);
            $payment->setState($state);
            $payment->save();
            $this->getLog('Payment ' . $state);
            $this->getKassaLogger()->sendHeka(['order-status.change.success']);
        } catch (Exception $e) {
            $this->getLog('Payment status update failed: ' . $e->getMessage(), 'error');
            $this->getKassaLogger()->sendHeka(['order-status.change.fail']);
        }
    }

    /**
     * @return YooKassaLoggerHelper
     */
    private function getKassaLogger(): YooKassaLoggerHelper
    {
        return $this->kassa->kassaLogger;
    }

    /**
     * @param string $message
     * @param string $type
     * @return void
     */
    private function getLog(string $message, string $type = 'info'): void
    {
        $this->kassa->log($message, $type);
    }
}
