<?php

/**
 * @author YooMoney <cms@yoomoney.ru>
 * @copyright © 2025 "YooMoney", NBСO LLC
 * @license  https://yoomoney.ru/doc.xml?id=527052
 */

namespace Drupal\yookassa\Oauth;

use Drupal;
use Drupal\yookassa\Plugin\Commerce\PaymentGateway\YooKassa;
use Exception;
use YooKassa\Client;

/**
 * Фабрика для получения единственного статического экземпляра клиента API Юkassa
 */
class YooKassaClientFactory
{
    public const YOOKASSA_ACCESS_TOKEN_KEY = 'access_token';
    public const CMS_NAME = 'drupal';
    public const MODULE_NAME = 'yoo_api_drupal10';

    /** @var Client|null Класс клиента API */
    private static ?Client $client = null;

    /**
     * Возвращает единственный инстанс клиента API Юkassa.
     *
     * @param array $config Конфигурация платежного шлюза
     *
     * @return Client
     * @throws Exception
     */
    public static function getYooKassaClient(array $config): Client
    {
        if (self::$client === null) {
            self::$client = self::createClient($config);
        }

        return self::$client;
    }

    /**
     * Создает и настраивает новый экземпляр клиента API.
     *
     * @param array $config Конфигурация платежного шлюза
     * @throws Exception
     */
    private static function createClient(array $config): Client
    {
        try {
            $apiClient = new Client();

            if (!empty($config[self::YOOKASSA_ACCESS_TOKEN_KEY])) {
                $apiClient->setAuthToken($config[self::YOOKASSA_ACCESS_TOKEN_KEY]);
            }

            self::configureClient($apiClient);

            return $apiClient;
        } catch (Exception $e) {
            throw new Exception('Failed to create YooKassa client: ' . $e->getMessage());
        }
    }

    /**
     * Устанавливает значение свойств для user-agent.
     *
     * @param Client $apiClient Класс клиента API
     *
     * @return void
     */
    private static function configureClient(Client $apiClient): void
    {
        $userAgent = $apiClient->getApiClient()->getUserAgent();
        $apiClient->setLogger(Drupal::logger('yookassa'));
        $userAgent->setCms(self::CMS_NAME, Drupal::VERSION);
        $userAgent->setModule(self::MODULE_NAME, YooKassa::YOOMONEY_MODULE_VERSION);
    }
}
