<?php

namespace Drupal\youtube_gallery\Commands;

use Drupal\Core\File\FileSystemInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use GuzzleHttp\ClientInterface;
use Symfony\Component\Filesystem\Filesystem;
use ZipArchive;

/**
 * Drush commands for the YouTube Gallery module.
 */
class YoutubeGalleryCommands extends DrushCommands {

  /**
   * The Google API PHP client zip URL.
   *
   * Using a constant on the command class instead of a global define.
   */
  private const GOOGLE_API_CLIENT_PHP_URL = 'https://github.com/google/google-api-php-client/archive/master.zip';

  /**
   * Constructs a new YoutubeGalleryCommands object.
   */
  public function __construct(
    protected ClientInterface $httpClient,
    protected FileSystemInterface $fileSystem
  ) {
    parent::__construct();
  }

  /**
   * Download the Google API PHP client libraries.
   *
   * @command ytg:libraries
   * @aliases ytg-libraries
   *
   * @param string $path
   *   Optional. A path where to install the Google libraries.
   *   If omitted, "libraries" under the Drupal root will be used.
   *
   * @arg path Optional path where to install the Google libraries (relative or absolute).
   *
   * @usage drush ytg:libraries
   *   Download the google-api-php-client into DRUPAL_ROOT/libraries.
   * @usage drush ytg:libraries web/libraries
   *   Download into the web/libraries directory.
   */
  public function downloadLibraries(string $path = 'libraries') : int {
    // Normalize the installation path: support relative or absolute.
    $root = \Drupal::root();
    if (!str_starts_with($path, '/') && !preg_match('@^[A-Za-z]:\\\\@', $path)) {
      // Treat as relative to DRUPAL_ROOT.
      $install_path = rtrim($root . '/' . $path, '/');
    }
    else {
      $install_path = rtrim($path, '/');
    }

    $filesystem = new Filesystem();

    // Ensure the directory exists.
    try {
      if (!is_dir($install_path)) {
        $filesystem->mkdir($install_path, 0755);
        $this->logger()->notice('Directory {path} was created.', ['path' => $install_path]);
      }
    }
    catch (\Throwable $e) {
      $this->logger()->error('Failed to create directory {path}: {msg}', [
        'path' => $install_path,
        'msg' => $e->getMessage(),
      ]);
      return self::EXIT_FAILURE;
    }

    // Download the zip file.
    $zip_path = $install_path . '/google-api-php-client.zip';

    try {
      $this->httpClient->get(self::GOOGLE_API_CLIENT_PHP_URL, [
        'sink' => $zip_path,
        'timeout' => 60,
      ]);
    }
    catch (\Throwable $e) {
      $this->logger()->error('Failed to download google-api-php-client from {url}: {msg}', [
        'url' => self::GOOGLE_API_CLIENT_PHP_URL,
        'msg' => $e->getMessage(),
      ]);
      return self::EXIT_FAILURE;
    }

    // Extract the archive.
    $zip = new ZipArchive();
    if ($zip->open($zip_path) !== TRUE) {
      $this->logger()->error('Unable to open downloaded archive at {path}.', ['path' => $zip_path]);
      return self::EXIT_FAILURE;
    }

    if (!$zip->extractTo($install_path)) {
      $zip->close();
      $this->logger()->error('Failed to extract archive into {path}.', ['path' => $install_path]);
      return self::EXIT_FAILURE;
    }
    $zip->close();

    // Clean up the zip file.
    try {
      $filesystem->remove($zip_path);
    }
    catch (\Throwable $e) {
      // Non-fatal.
      $this->logger()->warning('Could not remove temporary archive {path}: {msg}', [
        'path' => $zip_path,
        'msg' => $e->getMessage(),
      ]);
    }

    // Handle the directory names.
    $extracted_dir = $install_path . '/google-api-php-client-master';
    $target_dir = $install_path . '/google-api-php-client';

    try {
      if (is_dir($target_dir)) {
        $filesystem->remove($target_dir);
        $this->logger()->notice('Existing google-api-php-client library was deleted from {path}.', [
          'path' => $target_dir,
        ]);
      }

      if (is_dir($extracted_dir)) {
        $filesystem->rename($extracted_dir, $target_dir, true);
      }
    }
    catch (\Throwable $e) {
      $this->logger()->error('Failed to finalize google-api-php-client directory: {msg}', [
        'msg' => $e->getMessage(),
      ]);
      return self::EXIT_FAILURE;
    }

    if (is_dir($target_dir)) {
      $this->logger()->success('google-api-php-client library has been installed in {path}.', [
        'path' => $target_dir,
      ]);
      return self::EXIT_SUCCESS;
    }

    $this->logger()->error('Drush was unable to install the google-api-php-client library into {path}.', [
      'path' => $install_path,
    ]);
    return self::EXIT_FAILURE;
  }

}

