<?php

namespace Drupal\youtube_latest_block\Plugin\Block;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Drupal\Core\Routing\CurrentRouteMatch;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a 'Youtube Latest Block' block.
 *
 * @Block(
 *   id = "youtube_latest_block",
 *   admin_label = @Translation("Youtube Latest Block"),
 *   category = @Translation("Blocks")
 * )
 */
class YoutubeLatestBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The config object.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The route matcher.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a new OEmbedBlock instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config object.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $current_route_match
   *   The config object.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ConfigFactoryInterface $config_factory,
    CurrentRouteMatch $current_route_match,
    LoggerChannelFactoryInterface $logger_factory,
    RequestStack $request_stack,
    ClientInterface $http_client,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->currentRouteMatch = $current_route_match;
    $this->loggerFactory = $logger_factory;
    $this->requestStack = $request_stack;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('logger.factory'),
      $container->get('request_stack'),
      $container->get('http_client')
    );
  }

  /**
   * Gets the youtube videos via api.
   *
   * @param string $channel_id
   *   The channel id.
   * @param string $per_page
   *   The number of items per page.
   * @param string $pageToken
   *   The pagination token.
   */
  private function getYouTubeLatest($channel_id, $per_page, $pageToken = NULL) {
    $configFactory = $this->configFactory->get('youtube_latest_block.settings');
    $api_key = $configFactory->get('youtube_latest_block_api_key');

    $pageToken = $this->requestStack->getCurrentRequest()->query->all()['pageToken'] ?? NULL;

    // Cost 100 units.
    $url = "https://www.googleapis.com/youtube/v3/search?part=snippet&channelId={$channel_id}&order=date&maxResults={$per_page}&pageToken={$pageToken}&key={$api_key}";

    try {
      $response = $this->httpClient->get($url);
      $all_data = $response->getBody()->getContents();
      $data = json_decode($all_data, TRUE);

      return $data;
    }
    catch (RequestException $e) {
      // Handle API request errors.
      $this->loggerFactory->get('youtube_latest_block')->error('Error fetching YouTube videos: @message', ['@message' => $e->getMessage()]);
      return [];
    }
    catch (ClientException $e) {
      // Handle API request errors.
      $this->loggerFactory->get('youtube_latest_block')->error('Error fetching YouTube videos: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $config = $this->getConfiguration();

    $form['youtube_latest_block_channel_video'] = [
      '#type' => 'radios',
      '#title' => $this->t('Channel or Video Id'),
      '#options' => [
        'channel_id' => $this->t('Channel ID'),
        'video_id' => $this->t('Video ID'),
      ],
      '#default_value' => $config['youtube_latest_block_channel_video'] ?? 'channel_id',
      '#multiple' => FALSE,
    ];

    $form['youtube_latest_block_video_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Video Id'),
      '#default_value' => $config['youtube_latest_block_video_id'] ?? '',
      '#description' => $this->t('Enter the video id of the youtube channel.'),
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'video_id']],
        ],
      ],
    ];

    $form['youtube_latest_block_channel_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel Id'),
      '#default_value' => $config['youtube_latest_block_channel_id'] ?? '',
      '#description' => $this->t('Enter the channel id of the youtube channel.'),
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id']],
        ],
      ],
    ];

    $form['youtube_latest_block_if_multiple'] = [
      '#type' => 'radios',
      '#title' => $this->t('If single item or multiple'),
      '#options' => [
        'single' => $this->t('Single'),
        'multiple' => $this->t('Multiple'),
      ],
      '#default_value' => $config['youtube_latest_block_if_multiple'] ?? 'single',
      '#multiple' => FALSE,
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id']],
        ],
      ],
    ];

    $form['youtube_latest_block_show_records_data'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Show Records Data'),
      '#default_value' => $config['youtube_latest_block_show_records_data'] ?? '',
      '#description' => $this->t('Shows the total record and per page at the top of the listing.'),
      '#states' => [
        'visible' => [
          ':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id'],
          ':input[name="settings[youtube_latest_block_if_multiple]"]' => ['value' => 'multiple'],
        ],
      ],
    ];

    $form['youtube_latest_block_per_page'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Items Per Page'),
      '#default_value' => $config['youtube_latest_block_per_page'] ?? '',
      '#description' => $this->t('Enter the number of items per page.'),
      '#states' => [
        'visible' => [
          ':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id'],
          ':input[name="settings[youtube_latest_block_if_multiple]"]' => ['value' => 'multiple'],
        ],
      ],
    ];

    $form['youtube_latest_block_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Youtube Date'),
      '#default_value' => $config['youtube_latest_block_date'] ?? '',
      '#description' => $this->t('Render the published date provided by the channel.'),
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id']],
        ],
      ],
    ];

    $form['youtube_latest_block_date_placement'] = [
      '#type' => 'select',
      '#title' => $this->t('Place Date Over or Under Video'),
      '#options' => [
        'over' => $this->t('Over'),
        'under' => $this->t('Under'),
      ],
      '#default_value' => $config['youtube_latest_block_date_placement'] ?? 'over',
      '#multiple' => FALSE,
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_date]"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['youtube_latest_block_title'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Youtube Title'),
      '#default_value' => $config['youtube_latest_block_title'] ?? '',
      '#description' => $this->t('Render the title provided by the channel.'),
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id']],
        ],
      ],
    ];

    $form['youtube_latest_block_title_placement'] = [
      '#type' => 'select',
      '#title' => $this->t('Place Title Over or Under Video'),
      '#options' => [
        'over' => $this->t('Over'),
        'under' => $this->t('Under'),
      ],
      '#default_value' => $config['youtube_latest_block_title_placement'] ?? 'over',
      '#multiple' => FALSE,
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_title]"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['youtube_latest_block_description'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Youtube Description'),
      '#default_value' => $config['youtube_latest_block_description'] ?? '',
      '#description' => $this->t('Render the description provided by the channel.'),
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_channel_video]"]' => ['value' => 'channel_id']],
        ],
      ],
    ];

    $form['youtube_latest_block_description_placement'] = [
      '#type' => 'select',
      '#title' => $this->t('Place Description Over or Under Video'),
      '#options' => [
        'over' => $this->t('Over'),
        'under' => $this->t('Under'),
      ],
      '#default_value' => $config['youtube_latest_block_description_placement'] ?? 'over',
      '#multiple' => FALSE,
      '#states' => [
        'visible' => [
          [':input[name="settings[youtube_latest_block_description]"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->setConfigurationValue('youtube_latest_block_channel_video', $form_state->getValue('youtube_latest_block_channel_video'));
    $this->setConfigurationValue('youtube_latest_block_video_id', $form_state->getValue('youtube_latest_block_video_id'));
    $this->setConfigurationValue('youtube_latest_block_if_multiple', $form_state->getValue('youtube_latest_block_if_multiple'));
    $this->setConfigurationValue('youtube_latest_block_show_records_data', $form_state->getValue('youtube_latest_block_show_records_data'));
    $this->setConfigurationValue('youtube_latest_block_per_page', $form_state->getValue('youtube_latest_block_per_page'));
    $this->setConfigurationValue('youtube_latest_block_channel_id', $form_state->getValue('youtube_latest_block_channel_id'));
    $this->setConfigurationValue('youtube_latest_block_date', $form_state->getValue('youtube_latest_block_date'));
    $this->setConfigurationValue('youtube_latest_block_date_placement', $form_state->getValue('youtube_latest_block_date_placement'));
    $this->setConfigurationValue('youtube_latest_block_title', $form_state->getValue('youtube_latest_block_title'));
    $this->setConfigurationValue('youtube_latest_block_title_placement', $form_state->getValue('youtube_latest_block_title_placement'));
    $this->setConfigurationValue('youtube_latest_block_description', $form_state->getValue('youtube_latest_block_description'));
    $this->setConfigurationValue('youtube_latest_block_description_placement', $form_state->getValue('youtube_latest_block_description_placement'));
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];
    $config = $this->getConfiguration();

    $video_channel = $config['youtube_latest_block_channel_video'];
    $video_id = $config['youtube_latest_block_video_id'];
    $channel_id = $config['youtube_latest_block_channel_id'];

    $if_multiple = $config['youtube_latest_block_if_multiple'];
    $per_page = $config['youtube_latest_block_per_page'] != '' ? $config['youtube_latest_block_per_page'] : 10;
    $show_records_data = $config['youtube_latest_block_show_records_data'];

    $date = $config['youtube_latest_block_date'];
    $date_placement = $config['youtube_latest_block_date_placement'];
    $title = $config['youtube_latest_block_title'];
    $title_placement = $config['youtube_latest_block_title_placement'];
    $decription = $config['youtube_latest_block_description'];
    $decription_placement = $config['youtube_latest_block_description_placement'];

    if ($video_channel == 'channel_id') {
      $all_data = $this->getYouTubeLatest($channel_id, $per_page);
      $data = $all_data['items'][0];
      $video_id = $data["id"]["videoId"];
    }
    $video_url = "https://www.youtube.com/watch?v=" . $video_id;
    if (!empty($video_url)) {
      try {
        if ($video_channel == 'channel_id' && $if_multiple == 'single' && ($date_placement == 'over' || $title_placement == 'over' || $decription_placement == 'over')) {
          $content_over = [];
          $template_over = '<div class="youtube-latest-block-meta">';
          if ($date && $date_placement == 'over') {
            $template_over .= '<p class="youtube-latest-block-date">{{ date_over }}</p>';
            $content_over['date_over'] = date("m-d-Y h:i:s A", strtotime($data["snippet"]["publishedAt"]));
          }
          if ($title && $title_placement == 'over') {
            $template_over .= '<p class="youtube-latest-block-title">{{ title_over }}</p>';
            $content_over['title_over'] = $data["snippet"]["title"];
          }
          if ($decription && $decription_placement == 'over') {
            $template_over .= '<p class="youtube-latest-block-description">{{ description_over }}</p>';
            $content_over['description_over'] = $data["snippet"]["description"];
          }
          $template_over .= '</div>';

          $build['oembed_over_title'] = [
            '#type' => 'inline_template',
            '#template' => $template_over,
            '#context' => $content_over,
          ];
        }

        if ($if_multiple == 'multiple') {
          if ($show_records_data) {
            $pagination_info_template = '<p class="youtube-latest-block-pagination-total-results">{{ total_results }}</p>';
            $pagination_info_data['total_results'] = $all_data["pageInfo"]["totalResults"];
            $pagination_info_template .= '<p class="youtube-latest-block-pagination-results-per-page">{{ results_per_page }}</p>';
            $pagination_info_data['results_per_page'] = $all_data["pageInfo"]["resultsPerPage"];
            $build['pagination_info'] = [
              '#type' => 'inline_template',
              '#template' => $pagination_info_template,
              '#context' => $pagination_info_data,
            ];
          }

          // Get the current route.
          $current_route_name = $this->currentRouteMatch->getRouteName();

          $prevClasses = $nextClasses = [
            'button',
            'youtube-latest-block-pagination-button',
          ];
          $prevOptions = $nextOptions = [];

          // dump($all_data);
          if (isset($all_data["prevPageToken"]) && $all_data["prevPageToken"] != NULL) {
            $prevOptions[] = ['pageToken' => $all_data["prevPageToken"]];
          }
          else {
            $prevClasses[] = 'is-disabled';
          }

          $prev_page_link = Link::fromTextAndUrl('< Prev', Url::fromRoute($current_route_name, $prevOptions))->toRenderable();
          $prev_page_link['#attributes'] = ['class' => $prevClasses];
          $pagination_links_data['prev_page'] = $prev_page_link;

          if (isset($all_data["nextPageToken"]) && $all_data["nextPageToken"] != NULL) {
            $nextOptions[] = ['pageToken' => $all_data["nextPageToken"]];
          }
          else {
            $nextClasses[] = 'is-disabled';
          }

          $next_page_link = Link::fromTextAndUrl('Next >', Url::fromRoute($current_route_name, $nextOptions))->toRenderable();
          $next_page_link['#attributes'] = ['class' => $nextClasses];
          $pagination_links_data['next_page'] = $next_page_link;

          $pagination_links_template = '<div class="youtube-latest-block-pagination">{{ prev_page }} {{ next_page }}</div>';

          $build['pagination_over'] = [
            '#type' => 'inline_template',
            '#template' => $pagination_links_template,
            '#context' => $pagination_links_data,
            '#suffix' => '<div class="youtube-latest-block-multiple-container">',
          ];
          foreach ($all_data['items'] as $key => $video) {
            $build['oembed_content_' . $key] = [
              '#type' => 'inline_template',
              '#template' => '<div><iframe src="https://www.youtube.com/embed/{{ video_id }}" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe></div>',
              '#context' => [
                'video_id' => $video["id"]["videoId"],
              ],
            ];
          }
          $build['pagination_under'] = [
            '#type' => 'inline_template',
            '#template' => $pagination_links_template,
            '#context' => $pagination_links_data,
            '#prefix' => '</div>',
          ];
        }
        else {
          $build['oembed_content'] = [
            '#type' => 'inline_template',
            '#template' => '<div class="youtube-latest-block-container"><iframe src="https://www.youtube.com/embed/{{ video_id }}" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe></div>',
            '#context' => [
              'video_id' => $video_id,
            ],
          ];
        }

        if ($video_channel == 'channel_id' && $if_multiple == 'single' && ($date_placement == 'under' || $title_placement == 'under' || $decription_placement == 'under')) {
          $content_under = [];
          $template_under = '<div class="youtube-latest-block-meta">';
          if ($date && $date_placement == 'under') {
            $template_under .= '<p class="youtube-latest-block-date">{{ date_under }}</p>';
            $content_under['date_under'] = date("m-d-Y h:i:s A", strtotime($data["snippet"]["publishedAt"]));
          }
          if ($title && $title_placement == 'under') {
            $template_under .= '<p class="youtube-latest-block-title">{{ title_under }}</p>';
            $content_under['title_under'] = $data["snippet"]["title"];
          }
          if ($decription && $decription_placement == 'under') {
            $template_under .= '<p class="youtube-latest-block-description">{{ description_under }}</p>';
            $content_under['description_under'] = $data["snippet"]["description"];
          }
          $template_under .= '</div>';

          $build['oembed_under_title'] = [
            '#type' => 'inline_template',
            '#template' => $template_under,
            '#context' => $content_under,
          ];
        }
        $build['#attached']['library'][] = 'youtube_latest_block/block_styles';
      }
      catch (\Exception $e) {
        $build['oembed_content'] = [
          '#markup' => $this->t('Error embedding content: @message', ['@message' => $e->getMessage()]),
        ];
      }
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
