<?php

namespace Drupal\youtube_playlists;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Google\Service\YouTube;
use Google\Client;

/**
 * Youtube Integration to fetch content and update nodes in Drupal.
 */
class YoutubePlaylistsIntegration {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The Google Client.
   *
   * @var \Google\Client
   */
  protected $googleClient;

  /**
   * {@inheritDoc}
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, Client $google_client) {
    $this->entityTypeManager = $entity_type_manager;
    $this->googleClient = $google_client;
  }

  /**
   * Fetch Youtube Playlists and update nodes.
   */
  public function fetchYoutubeContent() {
    $client = $this->googleClient;
    $state = \Drupal::state();
    $youtube_api_key = $state->get('youtube_playlists.youtube_api_key');
    $youtube_channel_id = $state->get('youtube_playlists.youtube_channel_id');

    if (empty($youtube_api_key) || empty($youtube_channel_id)) {
      \Drupal::logger('youtube_integration')->error('API key or Channel ID is missing.');
      return;
    }

    $apiKey = $youtube_api_key;
    $client->setDeveloperKey($apiKey);
    $service = new YouTube($client);

    $playlists = [];
    $nextPageToken = NULL;

    try {
      do {
        $response = $service->playlists->listPlaylists('snippet', [
          'channelId' => $youtube_channel_id,
          'maxResults' => 50,
          'pageToken' => $nextPageToken,
        ]);

        $playlists = array_merge($playlists, $response->getItems());
        $nextPageToken = $response->getNextPageToken();
      } while ($nextPageToken);
    }
    catch (\Exception $e) {
      \Drupal::logger('youtube_integration')->error('Failed fetching playlists: @message', ['@message' => $e->getMessage()]);
      return;
    }

    foreach ($playlists as $playlist) {
      $playlistId = $playlist['id'];
      $title = $playlist['snippet']['title'];
      $thumbnailUrl = $playlist['snippet']['thumbnails']['high']['url'];
      $publishedAt = $playlist['snippet']['publishedAt'];
      $description = $playlist['snippet']['description'];

      $playlistItems = [];
      $nextPageToken = NULL;

      try {
        do {
          $response = $service->playlistItems->listPlaylistItems('snippet', [
            'playlistId' => $playlistId,
            'maxResults' => 50,
            'pageToken' => $nextPageToken,
          ]);

          $playlistItems = array_merge($playlistItems, $response->getItems());
          $nextPageToken = $response->getNextPageToken();
        } while ($nextPageToken);
      }
      catch (\Exception $e) {
        \Drupal::logger('youtube_integration')->error('Failed fetching playlist items: @message', ['@message' => $e->getMessage()]);
        // Skip processing this playlist.
        continue;
      }

      $totalDuration = 0;

      foreach ($playlistItems as $playlistItem) {
        $videoId = $playlistItem['snippet']['resourceId']['videoId'];

        try {
          $videoResponse = $service->videos->listVideos('contentDetails', ['id' => $videoId]);

          if (!empty($videoResponse->getItems())) {
            $videoDuration = $videoResponse->getItems()[0]['contentDetails']['duration'];
            $videoDurationSeconds = $this->youtubePlaylistsParseDurationToSeconds($videoDuration);
            $totalDuration += $videoDurationSeconds;
          }
        }
        catch (\Exception $e) {
          \Drupal::logger('youtube_integration')->error('Failed fetching video details: @message', ['@message' => $e->getMessage()]);
          // Skip processing this video.
          continue;
        }
      }

      $totalHours = $this->youtubePlaylistsSecondsToHoursMinutes($totalDuration);
      $total_episodes = count($playlistItems);
      $publishedAtFormatted = date('Y-m-d\TH:i:s', strtotime($publishedAt));
      $playlistUrl = 'https://www.youtube.com/playlist?list=' . $playlistId;
      $existing_nodes = $this->entityTypeManager
        ->getStorage('node')
        ->loadByProperties([
          'type' => 'youtube_playlists',
          'field_yp_playlist_youtube_link' => $playlistUrl,
        ]);
      if (!empty($existing_nodes)) {
        foreach ($existing_nodes as $node) {
          $node->setTitle($title);
          $node->set('field_yp_description', $description);
          $node->set('field_yp_episodes', $total_episodes);
          $node->set('field_yp_published_at', $publishedAtFormatted);
          $node->set('field_yp_thumbnail', $thumbnailUrl);
          $node->set('field_yp_playlist_youtube_link', $playlistUrl);
          $node->set('field_yp_playlist_length', $totalDuration);
          $node->set('field_yp_total_hours', $totalHours);
          $node->set('field_yp_playlist_id', $playlistId);
          try {
            $node->save();
            \Drupal::logger('youtube_integration')->info('Updated node for playlist ID: @playlistId', ['@playlistId' => $playlistId]);
          }
          catch (\Exception $e) {
            \Drupal::logger('youtube_integration')->error('Failed to update node for playlist ID: @playlistId. Error: @message', [
              '@playlistId' => $playlistId,
              '@message' => $e->getMessage(),
            ]);
          }
        }
      }
    }
  }

  /**
   * Function to parse duration string into seconds.
   */
  private function youtubePlaylistsParseDurationToSeconds($duration) {
    if (empty($duration)) {
      return 0;
    }

    // The pattern extracts hours, minutes, and seconds from a format such
    // as PT15H10M25S.
    preg_match('/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/', $duration, $matches);

    // Extract hours, minutes, and seconds.
    $hours = intval($matches[1] ?? 0);
    $minutes = intval($matches[2] ?? 0);
    $seconds = intval($matches[3] ?? 0);
    // Calculate total duration in seconds.
    $totalSeconds = ($hours * 3600) + ($minutes * 60) + $seconds;

    return $totalSeconds;
  }

  /**
   * Function to convert seconds to both hours and minutes.
   */
  private function youtubePlaylistsSecondsToHoursMinutes($seconds) {
    // Convert seconds to hours.
    $hours = floor($seconds / 3600);

    // Calculate remaining minutes.
    $remainderSeconds = $seconds % 3600;
    $minutes = round($remainderSeconds / 60);

    // Round minutes.
    if ($minutes >= 50) {
      $hours += 1;
      $minutes = 0;
    }

    // Construct the time format.
    $timeFormat = '';
    if ($hours > 0) {
      $timeFormat .= $hours == 1 ? '1 hour' : $hours . ' hours';
    }
    if ($minutes > 0) {
      $timeFormat .= ($hours > 0 ? ' ' : '') . $minutes . ' ' . ($minutes === 1 ? 'minute' : 'minutes');
    }

    // Return the formatted time.
    return $timeFormat;
  }

}
