<?php

namespace Drupal\yusaopeny_ymca360\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Drupal\yusaopeny_ymca360\Y360Client;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a configuration form for the YMCA360 integration.
 *
 * Allows administrators to:
 * - Configure API credentials for connecting to YMCA360.
 * - Select which YMCA360 schedules should be synchronized to the site.
 * - Set optional import limits.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Drupal State service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Module Handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $module_handler;

  /**
   * Y360Client service.
   *
   * @var \Drupal\yusaopeny_ymca360\Y360Client
   */
  protected $client;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
    ModuleHandlerInterface $module_handler,
    Y360Client $client,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->state = $state;
    $this->module_handler = $module_handler;
    $this->client = $client;
    $this->entityTypeManager = $entity_type_manager;

    parent::__construct($config_factory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('module_handler'),
      $container->get('yusaopeny_ymca360.y360_client'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'yusaopeny_ymca360_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['yusaopeny_ymca360.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('yusaopeny_ymca360.settings');
    $form['credentials'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('API Credentials'),
      '#tree' => TRUE,
    ];

    $form['credentials']['user'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User'),
      '#default_value' => $config->get('credentials.user'),
      '#required' => TRUE,
    ];
    $form['credentials']['password'] = [
      '#type' => 'password',
      '#title' => $this->t('Password'),
      '#description' => $this->t('Please reach out to YMCA 360 representative to obtain the API credentials.'),
      '#default_value' => $config->get('credentials.password'),
      '#attributes' => [
        'placeholder' => 'password',
        'value' => $config->get('credentials.password'),
      ],
    ];

    if ($schedule_options = $this->getScheduleOptions()) {
      $form['schedule'] = [
        '#type' => 'details',
        '#open' => TRUE,
        '#title' => $this->t('Schedules to sync'),
        '#description' => $this->t('Select YMCA360 schedules you want to be synced down to your Open Y website and specify program subcategories they correspond to.'),
        '#tree' => TRUE,
      ];

      $form['schedule']['schedules'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('YMCA 360 Schedule'),
          $this->t('Local Program Subcategory'),
        ],
        '#description' => $this->t('Tick the schedules to sync'),
        '#empty' => $this->t('No schedules available'),
      ];

      $schedule_mapping = $config->get('schedule.schedules') ?? [];
      $keys_config = array_keys($schedule_mapping);
      $keys_schedule = array_keys($schedule_options);
      $keys = array_unique(array_merge($keys_config, $keys_schedule));
      sort($keys);

      $node_storage = $this->entityTypeManager->getStorage('node');

      foreach ($keys as $id) {
        $label = $schedule_options[$id] ?? '';
        $category = NULL;
        if ($category_id = $schedule_mapping[$id]['subcategory'] ?? NULL) {
          $category = $node_storage->load($category_id);
        }

        $form['schedule']['schedules'][$id]['enable'] = [
          '#type' => 'checkbox',
          '#title' => "$label ($id)",
          '#default_value' => (bool) $schedule_mapping[$id]['enable'],
        ];

        $form['schedule']['schedules'][$id]['subcategory'] = [
          '#type' => 'entity_autocomplete',
          '#title' => $this->t('Program subcategory'),
          '#title_display' => 'invisible',
          '#target_type' => 'node',
          '#default_value' => $category,
          '#selection_settings' => [
            'target_bundles' => ['program_subcategory'],
          ],
          '#states' => [
            'required' => [
              '[name="schedule[schedules][' . $id . '][enable]"]' => ['checked' => TRUE],
            ],
          ],
        ];
      }
    }

    $form['limit'] = [
      '#type' => 'select',
      '#title' => $this->t('Import Limit'),
      '#required' => FALSE,
      '#options' => [
        0 => $this->t('No limit'),
        50 => $this->t('50 records'),
        100 => $this->t('100 records'),
        250 => $this->t('250 records'),
        500 => $this->t('500 records'),
        700 => $this->t('700 records'),
        1000 => $this->t('1000 records'),
        1500 => $this->t('1500 records'),
      ],
      '#default_value' => $config->get('limit'),
    ];

    $form['reservation_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Registration link title'),
      '#description' => $this->t('The title for the registration link shown on session pages. Default: REGISTER.'),
      '#default_value' => $config->get('reservation_text'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $creds = $form_state->getValue('credentials');
    $data = $this->client->verifyCredentials($creds);
    if (!$data) {
      $form_state->setErrorByName('credentials', $this->t('Credentials are not valid.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('yusaopeny_ymca360.settings')
      ->set('credentials', $form_state->getValue('credentials'))
      ->set('schedule', $form_state->getValue('schedule') ?? [])
      ->set('limit', $form_state->getValue('limit'))
      ->set('reservation_text', $form_state->getValue('reservation_text'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Dummy schedule options.
   *
   * @return string[]
   *   An associative array of schedule options;
   */
  private function getScheduleOptions(): array {
    $options = $this->client->getByScheduleFilter();
    $result = [];
    if (!empty($options)) {
      array_map(function ($item) use (&$result) {
        $result[$item['id']] = $item['label'];
      }, $options);
    }
    return $result;
  }

}
