<?php

namespace Drupal\zenwidgets\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

final class ZenWidgetsService {

  use StringTranslationTrait;

  /**
   * @var LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * @var ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * @param ConfigFactoryInterface $config_factory
   * @param LoggerInterface $logger
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerInterface $logger,
  ) {
    $this->config = $config_factory->get('zenwidgets.config');
    $this->logger = $logger;
  }

  /**
   * @return string
   */
  public function getApiDomain(): string {
    $domain = $this->config->get('domain') ?: 'https://www.zenwidgets.com';
    $domain = rtrim($domain, '/');
    return $domain;
  }

  /**
   * @return string
   */
  public function getScriptUrl(): string {
    return $this->getApiDomain() . '/scripts/index.js';
  }

  /**
   * @return string|null
   */
  public function getWebsiteId(): ?string {
    return $this->config->get('website_id');
  }

  /**
   * @return string|null
   */
  public function getAuthenticationToken(): ?string {
    return $this->config->get('authentication_token');
  }

  /**
   * @return string|null
   */
  public function getHost(): ?string {
    return $_SERVER['HTTP_HOST'] ?? NULL;
  }

  /**
   * @return array
   */
  public function getScriptRenderArray(): array {
    $script_location = $this->getScriptUrl();
    $website_id = $this->getWebsiteId();
    $website_token = $this->getAuthenticationToken();

    if (!$website_id || !$website_token) {
      $this->logger->error('Missing website ID and/or authentication token.');
      return [];
    }

    return [
      '#type' => 'html_tag',
      '#tag' => 'script',
      '#attributes' => [
        'async' => TRUE,
        'type' => 'module',
        'data-zen-website-id' => $website_id,
        'data-zen-website-token' => $website_token,
        'src' => $script_location,
      ],
    ];
  }

  /**
   * @param string $widget_id
   * @param string|null $instance_data
   *
   * @return array
   */
  public function getWidgetRenderArray(string $widget_id, ?string $instance_data = NULL): array {
    return [
      '#type' => 'html_tag',
      '#tag' => 'span',
      '#attributes' => [
        'data-zen-widget-id' => $widget_id,
        'data-zen-widget-data' => $instance_data,
      ],
    ];
  }

  /**
   * @return array|null
   */
  public function getWidgets(): ?array {
    $query = http_build_query([
      'website_id' => $this->getWebsiteId(),
      'website_token' => $this->getAuthenticationToken(),
      'language' => \Drupal::languageManager()->getCurrentLanguage()->getId(),
      'host' => $this->getHost(),
    ]);
    $url = $this->getApiDomain() . '/api/widgets?' . $query;

    /** @var \GuzzleHttp\ClientInterface $httpClient */
    $httpClient = \Drupal::service('http_client');
    try {
      $requestResult = $httpClient->request('GET', $url);
    }
    catch (GuzzleException $exception) {
      $this->logger->error('Failed to load widgets from @url (@error)', [
        '@url' => $url,
        '@error' => $exception->getMessage(),
      ]);
      return NULL;
    }

    $content = $requestResult->getBody()->getContents();

    $json = json_decode($content);
    return $json?->instances;
  }

}
