<?php

namespace Drupal\zenwidgets\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\zenwidgets\Service\ZenWidgetsService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * ZEN widget Drupal widget.
 */
#[FieldWidget(
  id: 'zenwidgets_widget_default',
  label: new TranslatableMarkup('Widget'),
  field_types: ['zenwidgets_widget'],
)]
class WidgetDefaultWidget extends WidgetBase {

  /**
   * Constructs a WidgetDefaultWidget object.
   *
   * @param string $plugin_id
   *   The plugin ID for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\zenwidgets\Service\ZenWidgetsService $widgetsService
   *   The ZEN widgets service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    protected ZenWidgetsService $widgetsService,
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $settings,
      $third_party_settings,
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('zenwidgets'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element['value'] =
      $element + [
        '#type' => 'select',
        '#default_value' => $items[$delta]->value ?? NULL,
        '#options' => $this->getWidgetOptions(),
        '#empty_option' => $this->t('- None -'),
      ];
    return $element;
  }

  /**
   * Returns the available widgets as an options list.
   *
   * @return array
   *   The available widgets as options.
   */
  private function getWidgetOptions(): array {
    $result = [];

    $availableWidgets = $this->widgetsService->getWidgets();
    if (!$availableWidgets) {
      return $result;
    }

    foreach ($availableWidgets as $instance) {
      $title = $instance->title ?? $this->t('- no title -');
      $result[$instance->id] = "$instance->type: $title ($instance->id)";
    }

    return $result;
  }

}
