<?php

namespace Drupal\zenwidgets\Plugin\Field\FieldFormatter;

use Drupal\zenwidgets\Service\ZenWidgetsService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Widget field formatter.
 */
#[FieldFormatter(
  id: 'zenwidgets_widget_default_formatter',
  label: new TranslatableMarkup('Widget default formatter'),
  field_types: [
    'zenwidgets_widget',
  ],
)]
final class WidgetDefaultFormatter extends FormatterBase {

  /**
   * ZEN widgets service.
   *
   * @var \Drupal\zenwidgets\Service\ZenWidgetsService
   *   The ZEN widgets service.
   */
  protected ZenWidgetsService $widgetsService;

  /**
   * Constructs a WidgetDefaultFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\zenwidgets\Service\ZenWidgetsService $widgets_service
   *   The ZEN widgets service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, ZenWidgetsService $widgets_service) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->widgetsService = $widgets_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('zenwidgets')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $summary[] = var_export($this->getSettings(), TRUE);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $elements[$delta] = [
        $this->widgetsService->getWidgetRenderArray($item->value, $item->instance_data),
      ];
    }

    return $elements;
  }

}
