<?php

namespace Drupal\zenwidgets\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * ZEN widgets service.
 */
final class ZenWidgetsService {

  use StringTranslationTrait;

  /**
   * Configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Constructor.
   *
   * @param GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger interface.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    protected ClientInterface $httpClient,
    protected LanguageManagerInterface $languageManager,
    protected LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->config = $config_factory->get('zenwidgets.config');
  }

  /**
   * Returns the ZEN widgets API domain.
   *
   * @return string
   *   The API domain.
   */
  public function getApiDomain(): string {
    $domain = $this->config->get('domain') ?: 'https://www.zenwidgets.com';
    $domain = rtrim($domain, '/');
    return $domain;
  }

  /**
   * Returns the script location.
   *
   * @return string
   *   The script location.
   */
  public function getScriptUrl(): string {
    return $this->getApiDomain() . '/scripts/index.js';
  }

  /**
   * Returns the configured website ID.
   *
   * @return string|null
   *   The website ID.
   */
  public function getWebsiteId(): ?string {
    return $this->config->get('website_id');
  }

  /**
   * Returns the configured authentication token.
   *
   * @return string|null
   *   The authentication token.
   */
  public function getAuthenticationToken(): ?string {
    return $this->config->get('authentication_token');
  }

  /**
   * Returns the current host.
   *
   * @return string|null
   *   The host.
   */
  public function getHost(): ?string {
    return $_SERVER['HTTP_HOST'] ?? NULL;
  }

  /**
   * Returns the render array that injects the script.
   *
   * @return array
   *   Render array.
   */
  public function getScriptRenderArray(): array {
    $script_location = $this->getScriptUrl();
    $website_id = $this->getWebsiteId();
    $website_token = $this->getAuthenticationToken();

    if (!$website_id || !$website_token) {
      $this->logger->error('Missing website ID and/or authentication token.');
      return [];
    }

    return [
      '#type' => 'html_tag',
      '#tag' => 'script',
      '#attributes' => [
        'async' => TRUE,
        'type' => 'module',
        'data-zen-website-id' => $website_id,
        'data-zen-website-token' => $website_token,
        'src' => $script_location,
      ],
    ];
  }

  /**
   * Returns the render array that injects the widget's embed code.
   *
   * @param string $widget_id
   *   The widget ID.
   * @param string|null $instance_data
   *   Optional instance data to affect the widgets behavior.
   *
   * @return array
   *   Render array.
   */
  public function getWidgetRenderArray(string $widget_id, ?string $instance_data = NULL): array {
    return [
      '#type' => 'html_tag',
      '#tag' => 'span',
      '#attributes' => [
        'data-zen-widget-id' => $widget_id,
        'data-zen-widget-data' => $instance_data,
      ],
    ];
  }

  /**
   * Returns all available widgets.
   *
   * @return array|null
   *   The available widgets.
   */
  public function getWidgets(): ?array {
    $query = http_build_query([
      'website_id' => $this->getWebsiteId(),
      'website_token' => $this->getAuthenticationToken(),
      'language' => $this->languageManager->getCurrentLanguage()->getId(),
      'host' => $this->getHost(),
    ]);
    $url = $this->getApiDomain() . '/api/widgets?' . $query;

    try {
      $requestResult = $this->httpClient->request('GET', $url);
    }
    catch (GuzzleException $exception) {
      $this->logger->error('Failed to load widgets from @url (@error)', [
        '@url' => $url,
        '@error' => $exception->getMessage(),
      ]);
      return NULL;
    }

    $content = $requestResult->getBody()->getContents();

    $json = json_decode($content);
    return $json?->instances;
  }

}
